#!/usr/bin/env python3
import os
import json
import zipfile
import requests
import base64
from datetime import datetime

# Configuration
API_URL = "https://webpoint.fr/myprotect/api/extension_receiver.php"
REFERENCE = "EXT-ADN-MAX-2025-V1"
EXTENSION_NAME = "ADN Player Max"

def create_extension_zip():
    """Create a ZIP file containing all extension files"""
    zip_filename = "adn_player_max.zip"
    
    # Files to include in the ZIP
    files_to_zip = [
        "manifest.json",
        "content.js",
        "content.css",
        "popup.html",
        "popup.js",
        "README.md",
        "icon16.png",
        "icon32.png",
        "icon48.png",
        "icon64.png",
        "icon128.png"
    ]
    
    with zipfile.ZipFile(zip_filename, 'w', zipfile.ZIP_DEFLATED) as zipf:
        for file in files_to_zip:
            if os.path.exists(file):
                zipf.write(file, file)
                print(f"✓ Added {file} to ZIP")
            else:
                print(f"⚠ Warning: {file} not found")
    
    print(f"\n✓ Created {zip_filename}")
    return zip_filename

def read_file_content(filename):
    """Read file content and return as string"""
    with open(filename, 'r', encoding='utf-8') as f:
        return f.read()

def create_prestashop_product():
    """Create PrestaShop product via API"""
    
    # Create the ZIP file
    zip_filename = create_extension_zip()
    
    # Read description files
    description_fr = read_file_content("description_fr.html")
    description_en = read_file_content("description_en.html")
    
    # Read logo file and encode to base64
    with open("logo_adn_player_max.svg", 'rb') as f:
        logo_base64 = base64.b64encode(f.read()).decode('utf-8')
    
    # Read ZIP file and encode to base64
    with open(zip_filename, 'rb') as f:
        zip_base64 = base64.b64encode(f.read()).decode('utf-8')
    
    # Prepare the data
    product_data = {
        "reference": REFERENCE,
        "name": {
            "fr": "ADN Player Max - Extension Chrome Gratuite",
            "en": "ADN Player Max - Free Chrome Extension"
        },
        "description": {
            "fr": description_fr,
            "en": description_en
        },
        "description_short": {
            "fr": "Maximisez instantanément le lecteur vidéo ADN pour une expérience anime immersive ! Extension 100% GRATUITE.",
            "en": "Instantly maximize ADN video player for an immersive anime experience! 100% FREE extension."
        },
        "price": "0.00",
        "categories": ["Extensions Chrome", "Streaming", "Video", "Anime"],
        "tags": {
            "fr": ["adn", "animation digital network", "lecteur video", "maximiser", "plein ecran", "anime", "streaming", "chrome", "extension gratuite"],
            "en": ["adn", "animation digital network", "video player", "maximize", "fullscreen", "anime", "streaming", "chrome", "free extension"]
        },
        "meta_title": {
            "fr": "ADN Player Max - Extension Chrome Gratuite pour Maximiser ADN",
            "en": "ADN Player Max - Free Chrome Extension to Maximize ADN"
        },
        "meta_description": {
            "fr": "Extension Chrome gratuite pour maximiser le lecteur vidéo ADN. Profitez de vos animes en grand format ! Installation en 1 clic.",
            "en": "Free Chrome extension to maximize ADN video player. Enjoy your anime in large format! 1-click installation."
        },
        "meta_keywords": {
            "fr": "adn,animation digital network,lecteur video,maximiser,plein ecran,anime,streaming",
            "en": "adn,animation digital network,video player,maximize,fullscreen,anime,streaming"
        },
        "features": {
            "fr": [
                {"name": "Compatibilité", "value": "Chrome, Edge, Brave, Opera"},
                {"name": "Taille", "value": "< 100KB"},
                {"name": "Prix", "value": "GRATUIT"},
                {"name": "Installation", "value": "1 clic"},
                {"name": "Permissions", "value": "Minimales"}
            ],
            "en": [
                {"name": "Compatibility", "value": "Chrome, Edge, Brave, Opera"},
                {"name": "Size", "value": "< 100KB"},
                {"name": "Price", "value": "FREE"},
                {"name": "Installation", "value": "1 click"},
                {"name": "Permissions", "value": "Minimal"}
            ]
        },
        "logo": {
            "filename": "logo_adn_player_max.svg",
            "content": logo_base64,
            "mime_type": "image/svg+xml"
        },
        "download_file": {
            "filename": zip_filename,
            "content": zip_base64,
            "mime_type": "application/zip"
        },
        "active": True,
        "visibility": "both",
        "condition": "new",
        "show_price": True,
        "available_for_order": True,
        "minimal_quantity": 1,
        "redirect_type": "404",
        "timestamp": datetime.now().isoformat()
    }
    
    # Send to API
    print("\n📡 Sending product to PrestaShop API...")
    
    try:
        response = requests.post(
            API_URL,
            json=product_data,
            headers={
                'Content-Type': 'application/json',
                'User-Agent': 'ADN-Player-Max-Creator/1.0'
            },
            timeout=30
        )
        
        if response.status_code == 200:
            result = response.json()
            print("\n✅ SUCCESS! Product created successfully!")
            print(f"Response: {json.dumps(result, indent=2)}")
        else:
            print(f"\n❌ Error: HTTP {response.status_code}")
            print(f"Response: {response.text}")
            
    except requests.exceptions.RequestException as e:
        print(f"\n❌ Request error: {str(e)}")
    except Exception as e:
        print(f"\n❌ Unexpected error: {str(e)}")
    
    # Clean up
    if os.path.exists(zip_filename):
        os.remove(zip_filename)
        print(f"\n🧹 Cleaned up {zip_filename}")

if __name__ == "__main__":
    print("🚀 ADN Player Max - PrestaShop Product Creator")
    print("=" * 50)
    
    # Change to script directory
    script_dir = os.path.dirname(os.path.abspath(__file__))
    os.chdir(script_dir)
    
    create_prestashop_product()
    print("\n✨ Process completed!")