const fs = require('fs');
const https = require('https');
const { exec } = require('child_process');
const path = require('path');

console.log('🚀 ADN Player Max - PrestaShop Product Creator');
console.log('='.repeat(50));

// Function to convert file to base64
function fileToBase64(filePath) {
    const buffer = fs.readFileSync(filePath);
    return buffer.toString('base64');
}

// Function to create ZIP
function createZip(callback) {
    console.log('📦 Creating extension ZIP...');
    
    const files = [
        'manifest.json', 'content.js', 'content.css', 
        'popup.html', 'popup.js', 'README.md',
        'icon16.png', 'icon32.png', 'icon48.png', 'icon64.png', 'icon128.png'
    ].filter(f => fs.existsSync(f)).join(' ');
    
    exec(`zip -r adn_player_max.zip ${files}`, (error) => {
        if (error) {
            console.error('Error creating ZIP:', error);
            // Try alternative method
            const archiver = require('archiver');
            const output = fs.createWriteStream('adn_player_max.zip');
            const archive = archiver('zip', { zlib: { level: 9 } });
            
            archive.pipe(output);
            files.split(' ').forEach(file => {
                archive.file(file, { name: file });
            });
            archive.finalize();
            
            output.on('close', callback);
        } else {
            console.log('✓ ZIP created successfully');
            callback();
        }
    });
}

// Main function
function sendToAPI() {
    try {
        // Read descriptions
        const descFr = fs.readFileSync('description_fr.html', 'utf8');
        const descEn = fs.readFileSync('description_en.html', 'utf8');
        
        // Convert files to base64
        const logoBase64 = fileToBase64('logo_adn_player_max.svg');
        const zipBase64 = fileToBase64('adn_player_max.zip');
        
        // Prepare data
        const productData = {
            reference: "EXT-ADN-MAX-2025-V1",
            name: {
                fr: "ADN Player Max - Extension Chrome Gratuite",
                en: "ADN Player Max - Free Chrome Extension"
            },
            description: {
                fr: descFr,
                en: descEn
            },
            description_short: {
                fr: "Maximisez instantanément le lecteur vidéo ADN pour une expérience anime immersive ! Extension 100% GRATUITE.",
                en: "Instantly maximize ADN video player for an immersive anime experience! 100% FREE extension."
            },
            price: "0.00",
            categories: ["Extensions Chrome", "Streaming", "Video", "Anime"],
            tags: {
                fr: ["adn", "animation digital network", "lecteur video", "maximiser", "plein ecran", "anime", "streaming", "chrome", "extension gratuite"],
                en: ["adn", "animation digital network", "video player", "maximize", "fullscreen", "anime", "streaming", "chrome", "free extension"]
            },
            meta_title: {
                fr: "ADN Player Max - Extension Chrome Gratuite pour Maximiser ADN",
                en: "ADN Player Max - Free Chrome Extension to Maximize ADN"
            },
            meta_description: {
                fr: "Extension Chrome gratuite pour maximiser le lecteur vidéo ADN. Profitez de vos animes en grand format ! Installation en 1 clic.",
                en: "Free Chrome extension to maximize ADN video player. Enjoy your anime in large format! 1-click installation."
            },
            meta_keywords: {
                fr: "adn,animation digital network,lecteur video,maximiser,plein ecran,anime,streaming",
                en: "adn,animation digital network,video player,maximize,fullscreen,anime,streaming"
            },
            features: {
                fr: [
                    {name: "Compatibilité", value: "Chrome, Edge, Brave, Opera"},
                    {name: "Taille", value: "< 100KB"},
                    {name: "Prix", value: "GRATUIT"},
                    {name: "Installation", value: "1 clic"},
                    {name: "Permissions", value: "Minimales"}
                ],
                en: [
                    {name: "Compatibility", value: "Chrome, Edge, Brave, Opera"},
                    {name: "Size", value: "< 100KB"},
                    {name: "Price", value: "FREE"},
                    {name: "Installation", value: "1 click"},
                    {name: "Permissions", value: "Minimal"}
                ]
            },
            logo: {
                filename: "logo_adn_player_max.svg",
                content: logoBase64,
                mime_type: "image/svg+xml"
            },
            download_file: {
                filename: "adn_player_max.zip",
                content: zipBase64,
                mime_type: "application/zip"
            },
            active: true,
            visibility: "both",
            condition: "new",
            show_price: true,
            available_for_order: true,
            minimal_quantity: 1,
            redirect_type: "404",
            timestamp: new Date().toISOString()
        };
        
        // Convert to JSON
        const jsonData = JSON.stringify(productData);
        
        console.log('\n📡 Sending to PrestaShop API...');
        console.log(`Data size: ${(jsonData.length / 1024 / 1024).toFixed(2)} MB`);
        
        // Parse URL
        const url = new URL('https://webpoint.fr/myprotect/api/extension_receiver.php');
        
        // Prepare request options
        const options = {
            hostname: url.hostname,
            port: 443,
            path: url.pathname,
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Content-Length': Buffer.byteLength(jsonData),
                'User-Agent': 'ADN-Player-Max-Creator/1.0'
            }
        };
        
        // Make request
        const req = https.request(options, (res) => {
            let data = '';
            
            res.on('data', (chunk) => {
                data += chunk;
            });
            
            res.on('end', () => {
                console.log(`\n✅ Response received! Status: ${res.statusCode}`);
                console.log('Response:', data);
                
                // Cleanup
                if (fs.existsSync('adn_player_max.zip')) {
                    fs.unlinkSync('adn_player_max.zip');
                    console.log('\n🧹 Cleaned up temporary files');
                }
                
                console.log('\n✨ Process completed!');
            });
        });
        
        req.on('error', (error) => {
            console.error('❌ Request error:', error);
        });
        
        // Send data
        req.write(jsonData);
        req.end();
        
    } catch (error) {
        console.error('❌ Error:', error);
    }
}

// Start process
createZip(() => {
    console.log('\n📤 Preparing to send data...');
    sendToAPI();
});