/**
 * Script préchargé dans le contenu de l'iframe
 * Modifie le localStorage avant le chargement du jeu pour forcer la langue française
 */

(function() {
    logger.log('Preload script exécuté - Initialisation des paramètres de langue');
    
    // Forcer la langue française dans localStorage avant le chargement du jeu
    try {
        // Vérifier si le localStorage existe déjà
        const persistedState = localStorage.getItem('persist:root');
        if (persistedState) {
            logger.log('État localStorage trouvé, mise à jour de la langue');
            
            try {
                const state = JSON.parse(persistedState);
                
                // Mettre à jour la langue (plusieurs formats possibles)
                if (state.language) {
                    // Format 1: "en" (chaîne simple)
                    if (state.language === '"en"' || state.language === '"fr"') {
                        state.language = '"fr"';
                    } 
                    // Format 2: {"state":"fr"} ou autre objet
                    else {
                        try {
                            const languageState = JSON.parse(state.language);
                            // Plusieurs possibilités selon la structure
                            if (typeof languageState === 'string') {
                                state.language = '"fr"';
                            } else if (languageState.state) {
                                languageState.state = 'fr';
                                state.language = JSON.stringify(languageState);
                            } else if (languageState.value) {
                                languageState.value = 'fr';
                                state.language = JSON.stringify(languageState);
                            } else {
                                // Format inconnu, essayer de remplacer la valeur directement
                                state.language = '"fr"';
                            }
                        } catch (e) {
                            // Format inconnu, remplacer directement
                            state.language = '"fr"';
                        }
                    }
                    
                    // Sauvegarder les modifications
                    localStorage.setItem('persist:root', JSON.stringify(state));
                    logger.log('Langue française forcée dans localStorage');
                }
            } catch (e) {
                logger.error('Erreur lors de la mise à jour de la langue:', e);
            }
        } else {
            logger.log('Aucun état localStorage trouvé, création d\'un état initial');
            
            // Créer un état initial avec la langue française
            const initialState = {
                language: '"fr"'
            };
            
            localStorage.setItem('persist:root', JSON.stringify(initialState));
            logger.log('État initial créé avec langue française');
        }
    } catch (e) {
        logger.error('Erreur lors de la manipulation du localStorage:', e);
    }
    
    // Intercepter Redux avant son chargement
    const originalDispatch = window.dispatch;
    window.reduxInitialized = false;
    
    // Fonction pour injecter un intercepteur dans Redux
    function injectReduxInterceptor() {
        logger.log('Tentative d\'injection d\'un intercepteur Redux');
        
        // Vérifier toutes les 500ms jusqu'à trouver Redux
        const interval = setInterval(() => {
            if (window.reduxInitialized) {
                clearInterval(interval);
                return;
            }
            
            for (let key in window) {
                if (window[key] && window[key].getState && typeof window[key].getState === 'function') {
                    logger.log('Store Redux trouvé');
                    window.reduxInitialized = true;
                    
                    const originalGetState = window[key].getState;
                    window[key].getState = function() {
                        const state = originalGetState.apply(this, arguments);
                        
                        // Modifier l'état pour forcer la langue française
                        if (state && state.language) {
                            // Ne rien faire si déjà en français
                            if (state.language === 'fr' || (state.language.value && state.language.value === 'fr')) {
                                return state;
                            }
                            
                            logger.log('Modif de l\'état Redux pour langue FR');
                            
                            // Dispatcher l'action pour changer la langue
                            if (window[key].dispatch) {
                                window[key].dispatch({ 
                                    type: 'language/setLanguage', 
                                    payload: { value: 'fr' }
                                });
                            }
                        }
                        
                        return state;
                    };
                    
                    clearInterval(interval);
                    break;
                }
            }
        }, 500);
        
        // Arrêter après 10 secondes si Redux n'est pas trouvé
        setTimeout(() => {
            clearInterval(interval);
            logger.log('Timeout: Redux n\'a pas été trouvé après 10 secondes');
        }, 10000);
    }
    
    // Exécuter l'interception dès que possible
    injectReduxInterceptor();
    
    // Également écouter l'événement DOMContentLoaded
    document.addEventListener('DOMContentLoaded', function() {
        logger.log('DOM chargé, initialisation des paramètres de langue');
        
        // Attendre un court délai puis forcer la langue française dans l'interface
        setTimeout(() => {
            const langSelector = document.querySelector('.react-select__single-value');
            if (langSelector && langSelector.textContent.includes('English')) {
                logger.log('Sélecteur de langue trouvé en anglais, passage en français');
                
                // Trouver le conteneur parent et cliquer pour ouvrir le menu
                const container = langSelector.closest('.language-selector') || 
                                langSelector.closest('.react-select') ||
                                langSelector.parentElement.parentElement;
                
                if (container) {
                    // Simuler un clic pour ouvrir le menu
                    container.click();
                    
                    // Petit délai puis chercher l'option française
                    setTimeout(() => {
                        // Rechercher toutes les options du menu
                        const options = Array.from(document.querySelectorAll('.react-select__option'));
                        
                        // Chercher l'option française
                        const frenchOption = options.find(option => 
                            option.textContent.includes('Français') || 
                            option.textContent.includes('French'));
                        
                        if (frenchOption) {
                            // Cliquer sur l'option française
                            frenchOption.click();
                            logger.log('Langue française sélectionnée via interface');
                        }
                    }, 200);
                }
            }
        }, 1000);
    });
})();